--------------------------------------------------------------------------------
-- 
-- CTU CAN FD IP Core 
-- Copyright (C) 2021-present Ondrej Ille
-- 
-- Permission is hereby granted, free of charge, to any person obtaining a copy
-- of this VHDL component and associated documentation files (the "Component"),
-- to use, copy, modify, merge, publish, distribute the Component for
-- educational, research, evaluation, self-interest purposes. Using the
-- Component for commercial purposes is forbidden unless previously agreed with
-- Copyright holder.
-- 
-- The above copyright notice and this permission notice shall be included in
-- all copies or substantial portions of the Component.
-- 
-- THE COMPONENT IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
-- IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
-- FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
-- AUTHORS OR COPYRIGHTHOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
-- LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
-- FROM, OUT OF OR IN CONNECTION WITH THE COMPONENT OR THE USE OR OTHER DEALINGS
-- IN THE COMPONENT.
-- 
-- The CAN protocol is developed by Robert Bosch GmbH and protected by patents.
-- Anybody who wants to implement this IP core on silicon has to obtain a CAN
-- protocol license from Bosch.
-- 
-- -------------------------------------------------------------------------------
-- 
-- CTU CAN FD IP Core 
-- Copyright (C) 2015-2020 MIT License
-- 
-- Authors:
--     Ondrej Ille <ondrej.ille@gmail.com>
--     Martin Jerabek <martin.jerabek01@gmail.com>
-- 
-- Project advisors: 
-- 	Jiri Novak <jnovak@fel.cvut.cz>
-- 	Pavel Pisa <pisa@cmp.felk.cvut.cz>
-- 
-- Department of Measurement         (http://meas.fel.cvut.cz/)
-- Faculty of Electrical Engineering (http://www.fel.cvut.cz)
-- Czech Technical University        (http://www.cvut.cz/)
-- 
-- Permission is hereby granted, free of charge, to any person obtaining a copy
-- of this VHDL component and associated documentation files (the "Component"),
-- to deal in the Component without restriction, including without limitation
-- the rights to use, copy, modify, merge, publish, distribute, sublicense,
-- and/or sell copies of the Component, and to permit persons to whom the
-- Component is furnished to do so, subject to the following conditions:
-- 
-- The above copyright notice and this permission notice shall be included in
-- all copies or substantial portions of the Component.
-- 
-- THE COMPONENT IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
-- IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
-- FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
-- AUTHORS OR COPYRIGHTHOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
-- LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
-- FROM, OUT OF OR IN CONNECTION WITH THE COMPONENT OR THE USE OR OTHER DEALINGS
-- IN THE COMPONENT.
-- 
-- The CAN protocol is developed by Robert Bosch GmbH and protected by patents.
-- Anybody who wants to implement this IP core on silicon has to obtain a CAN
-- protocol license from Bosch.
-- 
--------------------------------------------------------------------------------

--------------------------------------------------------------------------------
-- Module:
--  Memory registers
-- 
-- Purpose:
--  Configuration and Status registers are implemented here. Access over 32 bit,
--  Avalon compatible interface. Write in the same clock cycle, read data are
--  returned in next clock cycle. Driving bus is created here. Memory registers
--  are generated by Register Map Generation Tool.  
--------------------------------------------------------------------------------

Library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.ALL;

Library ctu_can_fd_rtl;
use ctu_can_fd_rtl.id_transfer_pkg.all;
use ctu_can_fd_rtl.can_constants_pkg.all;

use ctu_can_fd_rtl.can_types_pkg.all;
use ctu_can_fd_rtl.drv_stat_pkg.all;
use ctu_can_fd_rtl.unary_ops_pkg.all;

use ctu_can_fd_rtl.CAN_FD_register_map.all;
use ctu_can_fd_rtl.CAN_FD_frame_format.all;

use ctu_can_fd_rtl.can_registers_pkg.all;

entity memory_registers is
    generic(
        -- Support Filter A
        G_SUP_FILTA         : boolean                         := true;

        -- Support Filter B
        G_SUP_FILTB         : boolean                         := true;

        -- Support Filter C
        G_SUP_FILTC         : boolean                         := true;

        -- Support Range Filter
        G_SUP_RANGE         : boolean                         := true;

        -- Support Test registers
        G_SUP_TEST_REGISTERS: boolean                         := true;

        -- Support Traffic counters
        G_SUP_TRAFFIC_CTRS  : boolean                         := true;

        -- Number of TXT Buffers
        G_TXT_BUFFER_COUNT  : natural range 2 to 8            := 4;

        -- Number of Interrupts
        G_INT_COUNT         : natural                         := 12;

        -- Width (number of bits) in transceiver delay measurement counter
        G_TRV_CTR_WIDTH     : natural                         := 7;

        -- Number of active timestamp bits
        G_TS_BITS           : natural range 0 to 63           := 63;

        -- DEVICE_ID (read from register)
        G_DEVICE_ID         : std_logic_vector(15 downto 0)   := x"CAFD";

        -- MINOR Design version
        G_VERSION_MINOR     : std_logic_vector(7 downto 0)    := x"01";

        -- MAJOR Design version
        G_VERSION_MAJOR     : std_logic_vector(7 downto 0)    := x"02";
        
        -- Technology type
        G_TECHNOLOGY        : natural                         := C_TECH_ASIC
    );
    port(
        ------------------------------------------------------------------------
        -- Clock and Reset
        ------------------------------------------------------------------------
        -- System clock
        clk_sys              :in   std_logic;
        
        -- Asynchronous reset        
        res_n                :in   std_logic;
        
        -- Soft reset (Input reset + Software Reset)
        res_soft_n           :out  std_logic;

        -- Core Reset (Input reset + Software Reset + Active when disabled)
        res_core_n           :out  std_logic;

        -----------------------------------------------------------------------
        -- DFT support
        -----------------------------------------------------------------------
        scan_enable          :in   std_logic;

        ------------------------------------------------------------------------
        -- Memory Interface
        ------------------------------------------------------------------------
        -- Data input
        data_in              :in   std_logic_vector(31 downto 0);
        
        -- Data output
        data_out             :out  std_logic_vector(31 downto 0);
        
        -- Address
        adress               :in   std_logic_vector(15 downto 0);
        
        -- Chip Select
        scs                  :in   std_logic;
        
        -- Read
        srd                  :in   std_logic;
        
        -- Write
        swr                  :in   std_logic;
        
        -- Byte enable
        sbe                  :in   std_logic_vector(3 downto 0);
        
        -- Timestamp input
        timestamp            :in   std_logic_vector(63 downto 0);
        
        ------------------------------------------------------------------------
        -- Buses to/from rest of CTU CAN FD
        ------------------------------------------------------------------------
        -- Driving Bus
        drv_bus              :out  std_logic_vector(1023 downto 0);
        
        -- Status Bus
        stat_bus             :in   std_logic_vector(511 downto 0);

        ------------------------------------------------------------------------
        -- Manufacturing testability
        ------------------------------------------------------------------------
        -- Test registers outputs
        test_registers_out   :out  test_registers_out_t;
        
        -- RX buffer test data in
        tst_rdata_rx_buf     :in   std_logic_vector(31 downto 0);
        
        -- TXT buffers test data input
        tst_rdata_txt_bufs   :in   t_txt_bufs_output(G_TXT_BUFFER_COUNT - 1 downto 0);

        ------------------------------------------------------------------------
        -- RX Buffer Interface
        ------------------------------------------------------------------------
        -- RX Buffer data output
        rx_read_buff         :in   std_logic_vector(31 downto 0);

        -- Size of RX buffer (in words)
        rx_buf_size          :in   std_logic_vector(12 downto 0);

        -- RX Buffer is full
        rx_full              :in   std_logic;

        -- RX Buffer is empty
        rx_empty             :in   std_logic;

        -- Number of frames in RX buffer
        rx_frame_count       :in   std_logic_vector(10 downto 0);

        -- Number of free 32 bit words
        rx_mem_free          :in   std_logic_vector(12 downto 0);

        -- Position of read pointer
        rx_read_pointer      :in   std_logic_vector(11 downto 0);

        -- Position of write pointer
        rx_write_pointer     :in   std_logic_vector(11 downto 0);
            
        -- Data overrun Flag
        rx_data_overrun      :in   std_logic;
        
        -- Middle of frame indication
        rx_mof               :in   std_logic;

        -- RX Buffer parity error flag
        rx_parity_error      :in   std_logic;

        ------------------------------------------------------------------------
        -- Interface to TXT Buffers
        ------------------------------------------------------------------------
        -- TXT Buffer RAM - Data input
        txtb_port_a_data     :out  std_logic_vector(31 downto 0);
        
        -- TXT Buffer RAM - Address
        txtb_port_a_address  :out  std_logic_vector(4 downto 0);
        
        -- TXT Buffer chip select
        txtb_port_a_cs       :out  std_logic_vector(G_TXT_BUFFER_COUNT - 1 downto 0);
        
        -- TXT Buffer - byte enable
        txtb_port_a_be       :out  std_logic_vector(3 downto 0);

        -- TXT Buffer status
        txtb_state           :in   t_txt_bufs_state(G_TXT_BUFFER_COUNT - 1 downto 0);

        -- SW Commands to TXT Buffer
        txtb_sw_cmd          :out  t_txtb_sw_cmd;
        
        -- SW Command Index (Index in logic 1 means command is valid for TXT Buffer)          
        txtb_sw_cmd_index    :out  std_logic_vector(G_TXT_BUFFER_COUNT - 1 downto 0);
        
        -- TXT Buffer priorities
        txtb_prorities       :out  t_txt_bufs_priorities(G_TXT_BUFFER_COUNT - 1 downto 0);
        
        -- TXT Buffer bus-off behavior
        txt_buf_failed_bof   :out  std_logic;

        -- TXT Buffer Parity Error
        txtb_parity_error_valid :in std_logic_vector(G_TXT_BUFFER_COUNT - 1 downto 0);

        -- Parity Error in Backup buffer during TXT Buffer backup mode
        txtb_bb_parity_error :in   std_logic_vector(G_TXT_BUFFER_COUNT - 1 downto 0);

        ------------------------------------------------------------------------
        -- Bus synchroniser interface
        ------------------------------------------------------------------------
        -- Measured Transceiver Delay
        trv_delay            :in   std_logic_vector(G_TRV_CTR_WIDTH - 1 downto 0);
            
        ------------------------------------------------------------------------
        -- Interrrupt Interface
        ------------------------------------------------------------------------
        -- Interrupt vector
        int_vector           :in   std_logic_vector(G_INT_COUNT - 1 downto 0);
        
        -- Interrupt enable
        int_ena              :in   std_logic_vector(G_INT_COUNT - 1 downto 0);
        
        -- Interrupt mask
        int_mask             :in   std_logic_vector(G_INT_COUNT - 1 downto 0)
    );  
end entity;

architecture rtl of memory_registers is

    -- Generated register maps inputs/outputs
    signal control_registers_out    : control_registers_out_t;
    signal control_registers_in     : control_registers_in_t;

    signal test_registers_in        : test_registers_in_t;
    signal test_registers_out_i     : test_registers_out_t;
    
    -- Status register - combinational decoder
    signal status_comb              : std_logic_vector(31 downto 0);
    
    -- TXT buffer outputs - padded
    signal tst_rdata_txt_bufs_i     : t_txt_bufs_output(7 downto 0);

    -- Padding for interrupt read data
    constant INT_PAD_H_IND          : natural :=
        Control_registers_in.int_stat'length - G_INT_COUNT;

    constant INT_PADDING            : std_logic_vector(INT_PAD_H_IND -1 downto 0) :=
        (OTHERS => '0');

    -- Main chip select signal
    signal can_core_cs                : std_logic;

    -- Chip select signals for each memory sub-block
    signal control_registers_cs       : std_logic;
    signal control_registers_cs_reg   : std_logic;
    
    signal test_registers_cs          : std_logic;
    signal test_registers_cs_reg      : std_logic;
    
    -- Read data from generated register modules
    signal control_registers_rdata    : std_logic_vector(31 downto 0);
    signal test_registers_rdata       : std_logic_vector(31 downto 0);
   
    -- Fault confinement State Indication
    signal is_err_active          :     std_logic;
    signal is_err_passive         :     std_logic;
    signal is_bus_off             :     std_logic;
    
    -- Operation control state indication
    signal is_transmitter         :     std_logic;
    signal is_receiver            :     std_logic;
    signal is_idle                :     std_logic;
    
    -- Locks active
    signal reg_lock_1_active      :     std_logic;
    signal reg_lock_2_active      :     std_logic;
    
    -- Additional generated resets
    signal soft_res_d_n           :     std_logic;
    signal soft_res_q_n           :     std_logic;

    signal res_core_d_n           :     std_logic;
    
    signal ewl_padded             :     std_logic_vector(8 downto 0);

    -- Clock gating for register map
    signal control_regs_clk_en    :     std_logic;
    signal test_regs_clk_en       :     std_logic;
    
    signal clk_control_regs       :     std_logic;
    signal clk_test_regs          :     std_logic;

    -- RX buffer control signals
    signal rx_buf_mode              : std_logic;
    signal rx_move_cmd              : std_logic;

    signal ctr_pres_sel_q           : std_logic_vector(3 downto 0);

    signal tx_parity_error          : std_logic;
    signal tx_double_parity_error   : std_logic;

    signal drv_txbbm_ena            : std_logic;

    ---------------------------------------------------------------------------
    -- 
    ---------------------------------------------------------------------------
    function align_wrd_to_reg(
        reg_val         :   std_logic_vector;
        index           :   natural
    ) return std_logic is
    begin
        return reg_val(index mod reg_val'length);
    end function;

    ---------------------------------------------------------------------------
    --
    ---------------------------------------------------------------------------
    function align_wrd_to_reg(
        val             :   std_logic_vector;
        h_index         :   natural;
        l_index         :   natural
    ) return std_logic_vector is
        variable h_ind_mod  :  natural;
        variable l_ind_mod  :  natural;
    begin
        h_ind_mod := h_index mod val'length;
        l_ind_mod := l_index mod val'length;

        return val(h_ind_mod downto l_ind_mod); 
    end function;

    ---------------------------------------------------------------------------
    --
    ---------------------------------------------------------------------------    
    function align_reg_to_wrd(
        constant index          : in  natural;
        constant length         : in  natural
    ) return natural is
    begin
        return index mod length;
    end function;

begin
    
    ----------------------------------------------------------------------------
    -- Propagation of Avalon Data Bus to TXT Buffer RAM
    ----------------------------------------------------------------------------
    txtb_port_a_data      <= data_in;

    ----------------------------------------------------------------------------
    -- Since TX_DATA registers are in separate region, which is word aligned,
    -- it is enough to take the lowest bits to create the address offset.
    ----------------------------------------------------------------------------
    txtb_port_a_address   <= adress(6 downto 2);
  
    ---------------------------------------------------------------------------
    -- TXT Buffer RAMs chip select signals.
    ---------------------------------------------------------------------------
    txtb_port_a_cs_gen : for i in 0 to G_TXT_BUFFER_COUNT - 1 generate
        type tx_buff_addr_type is array (0 to 7) of
            std_logic_vector(3 downto 0);
        constant buf_addr : tx_buff_addr_type := (
            TX_BUFFER_1_BLOCK, TX_BUFFER_2_BLOCK,
            TX_BUFFER_3_BLOCK, TX_BUFFER_4_BLOCK,
            TX_BUFFER_5_BLOCK, TX_BUFFER_6_BLOCK,
            TX_BUFFER_7_BLOCK, TX_BUFFER_8_BLOCK
        );

    begin
        txtb_port_a_cs(i) <= '1' when ((adress(11 downto 8) = buf_addr(i)) and
                                        scs = '1' and swr = '1')
                                 else
                             '0';
    end generate txtb_port_a_cs_gen;

    txtb_port_a_be <= sbe;

    can_core_cs <= '1' when (scs = ACT_CSC) else
                   '0';

    ----------------------------------------------------------------------------
    -- Chip selects for register map blocks
    ----------------------------------------------------------------------------
    control_registers_cs <= '1' when (adress(11 downto 8) = CONTROL_REGISTERS_BLOCK)
                                      and (can_core_cs = '1')
                                else
                            '0';

    test_registers_cs <= '1' when (adress(11 downto 8) = TEST_REGISTERS_BLOCK)
                                   and (can_core_cs = '1')
                             else
                         '0';

    ----------------------------------------------------------------------------
    -- Registering control registers chip select
    ----------------------------------------------------------------------------
    chip_sel_reg_proc : process(res_n, clk_sys)
    begin
        if (res_n = '0') then
            control_registers_cs_reg  <= '0';
            test_registers_cs_reg <= '0';
        elsif (rising_edge(clk_sys)) then
            control_registers_cs_reg  <= control_registers_cs;
            test_registers_cs_reg <= test_registers_cs;
        end if;
    end process;

    ----------------------------------------------------------------------------
    -- Read data multiplexor. Use registered version of chip select signals
    -- since read data are returned one clock cycle later!
    ----------------------------------------------------------------------------
    data_out <= control_registers_rdata when (control_registers_cs_reg = '1') else
                   test_registers_rdata when (test_registers_cs_reg = '1') else
                        (OTHERS => '0');

    ----------------------------------------------------------------------------
    -- Clock gating - Ungate clocks for read or write.
    -- Note that write enable / read enable is still brought also to register
    -- map! This is for FPGA implementation where clock gate is transparent!
    ----------------------------------------------------------------------------
    control_regs_clk_en <= '1' when (srd = '1' or swr = '1') and
                                    (control_registers_cs = '1')
                               else
                           '0';

    test_regs_clk_en <= '1' when (srd = '1' or swr = '1') and
                                 (test_registers_cs = '1')
                            else
                        '0';

    clk_gate_control_regs_comp : entity ctu_can_fd_rtl.clk_gate
    generic map(
        G_TECHNOLOGY       => G_TECHNOLOGY
    )
    port map(
        clk_in             => clk_sys,
        clk_en             => control_regs_clk_en,
        scan_enable        => scan_enable,

        clk_out            => clk_control_regs
    );

    clk_gate_test_regs_comp : entity ctu_can_fd_rtl.clk_gate
    generic map(
        G_TECHNOLOGY       => G_TECHNOLOGY
    )
    port map(
        clk_in             => clk_sys,
        clk_en             => test_regs_clk_en,
        scan_enable        => scan_enable,

        clk_out            => clk_test_regs
    );

    ----------------------------------------------------------------------------
    -- Control registers instance
    ----------------------------------------------------------------------------
    control_registers_reg_map_comp : entity ctu_can_fd_rtl.control_registers_reg_map
    generic map(
        DATA_WIDTH            => 32,
        ADDRESS_WIDTH         => 16,
        REGISTERED_READ       => true,
        CLEAR_READ_DATA       => false,
        RESET_POLARITY        => '0',
        SUP_FILT_A            => G_SUP_FILTA,
        SUP_RANGE             => G_SUP_RANGE,
        SUP_FILT_C            => G_SUP_FILTC,
        SUP_FILT_B            => G_SUP_FILTB,
        SUP_TRAFFIC_CTRS      => G_SUP_TRAFFIC_CTRS
    )
    port map(
        clk_sys               => clk_control_regs,
        res_n                 => soft_res_q_n,
        address               => adress,
        w_data                => data_in,
        r_data                => control_registers_rdata,
        cs                    => control_registers_cs,
        read                  => srd,
        write                 => swr,
        be                    => sbe,
        lock_1                => reg_lock_1_active,
        lock_2                => reg_lock_2_active,
        control_registers_out => control_registers_out,
        control_registers_in  => control_registers_in
    );
    
    ----------------------------------------------------------------------------
    -- Test registers instance
    ----------------------------------------------------------------------------
    test_registers_gen_true : if (G_SUP_TEST_REGISTERS) generate
        test_registers_reg_map_comp : entity ctu_can_fd_rtl.test_registers_reg_map
        generic map (
            DATA_WIDTH          => 32,
            ADDRESS_WIDTH       => 16,
            REGISTERED_READ     => true,
            CLEAR_READ_DATA     => false,
            RESET_POLARITY      => '0'
        )
        port map(
            clk_sys             => clk_test_regs,
            res_n               => soft_res_q_n,
            address             => adress,
            w_data              => data_in,
            r_data              => test_registers_rdata,
            cs                  => test_registers_cs,
            read                => srd,
            write               => swr,
            be                  => sbe,
            lock_1              => reg_lock_1_active,
            lock_2              => reg_lock_2_active,
            test_registers_out  => test_registers_out_i,
            test_registers_in   => test_registers_in
        );

        -- Padding to full width of possible TXT Buffers
        txt_buf_test_data_padding_gen : for i in 0 to 7 generate
            
            txt_buf_padding_index_gen_true : if (i < G_TXT_BUFFER_COUNT) generate 
                tst_rdata_txt_bufs_i(i) <= tst_rdata_txt_bufs(i);
            end generate txt_buf_padding_index_gen_true;
            
            txt_buf_padding_index_gen_false : if (i >= G_TXT_BUFFER_COUNT) generate 
                tst_rdata_txt_bufs_i(i) <= (OTHERS => '0');
            end generate txt_buf_padding_index_gen_false;
            
        end generate;

        -- Select test read data from RX buffer and TXT buffers
        with test_registers_out_i.tst_dest(TST_MTGT_H downto TST_MTGT_L)
        select test_registers_in.tst_rdata <=
            tst_rdata_rx_buf when TMTGT_RXBUF,
            tst_rdata_txt_bufs_i(0) when TMTGT_TXTBUF1,
            tst_rdata_txt_bufs_i(1) when TMTGT_TXTBUF2,
            tst_rdata_txt_bufs_i(2) when TMTGT_TXTBUF3,
            tst_rdata_txt_bufs_i(3) when TMTGT_TXTBUF4,
            tst_rdata_txt_bufs_i(4) when TMTGT_TXTBUF5,
            tst_rdata_txt_bufs_i(5) when TMTGT_TXTBUF6,
            tst_rdata_txt_bufs_i(6) when TMTGT_TXTBUF7,
            tst_rdata_txt_bufs_i(7) when TMTGT_TXTBUF8,
                    (OTHERS => '0') when others;


        -- Indicate to register map that Test registers are present
        status_comb(STRGS_IND) <= '1';
        
    end generate test_registers_gen_true;
    
    test_registers_gen_false : if (not G_SUP_TEST_REGISTERS) generate
        test_registers_rdata <= (OTHERS => '0');
        test_registers_in.tst_rdata <= (OTHERS => '0');
        test_registers_out_i <= ((OTHERS => '0'), (OTHERS => '0'), (OTHERS => '0'));
        status_comb(STRGS_IND) <= '0';
        tst_rdata_txt_bufs_i <= (OTHERS => (OTHERS => '0'));
    end generate;

    test_registers_out <= test_registers_out_i;

    ----------------------------------------------------------------------------
    -- Several registers are locked and accessible only in Test mode!
    ----------------------------------------------------------------------------
    reg_lock_1_active <= not control_registers_out.mode(TSTM_IND);
    reg_lock_2_active <= control_registers_out.settings(ENA_IND mod 16);
    
    ----------------------------------------------------------------------------
    -- Two reset registers:
    --  1. Soft reset - Resets memory registers
    --  2. Core reset - Resets rest of the core when disabled.
    --
    -- Both need to be gated to inactive value in Scan mode since they reset
    -- other flops.
    ----------------------------------------------------------------------------

    -- Writing logic 1 to MODE[RST] causes Soft Reset
    soft_res_d_n <= not control_registers_out.mode(RST_IND);

    soft_rst_rst_reg_inst : entity ctu_can_fd_rtl.rst_reg
    generic map (
        G_RESET_POLARITY    => '0'
    )
    port map(
        -- Clock and Reset
        clk                 => clk_sys,
        arst                => res_n,

        -- Flip flop input / output
        d                   => soft_res_d_n,
        q                   => soft_res_q_n,

        -- Scan mode control
        scan_enable         => scan_enable
    );

    -- Reset of the rest of core is the same as soft reset, but
    -- it is also active when SETTINGS[ENA] = '0'. Thus disabled node
    -- has all of its logic in reset!
    res_core_d_n <= '0' when (control_registers_out.mode(RST_IND) = '1' or
                              control_registers_out.settings(ENA_IND mod 16) = '0')
                        else
                    '1';

    global_rst_rst_reg_inst : entity ctu_can_fd_rtl.rst_reg
    generic map (
        G_RESET_POLARITY    => '0'
    )
    port map(
        -- Clock and Reset
        clk                 => clk_sys,
        arst                => res_n,

        -- Flip flop input / output
        d                   => res_core_d_n,
        q                   => res_core_n,

        -- Scan mode control
        scan_enable         => scan_enable
    );

    ----------------------------------------------------------------------------
    -- Reset propagation to output
    ----------------------------------------------------------------------------
    res_soft_n <= soft_res_q_n;

    ----------------------------------------------------------------------------
    -- Extract Fault confinement state from Status Bus
    ----------------------------------------------------------------------------
    is_err_active <= stat_bus(STAT_IS_ERR_ACTIVE_INDEX);
    is_err_passive <= stat_bus(STAT_IS_ERR_PASSIVE_INDEX);
    is_bus_off <= stat_bus(STAT_IS_BUS_OFF_INDEX);
    
    ----------------------------------------------------------------------------
    -- Extract Operation Control information from Status Bus
    ----------------------------------------------------------------------------
    is_transmitter <= stat_bus(STAT_IS_TRANSMITTER_INDEX);
    is_receiver <= stat_bus(STAT_IS_RECEIVER_INDEX);
    is_idle <= stat_bus(STAT_IS_IDLE_INDEX);

    ---------------------------------------------------------------------------
    -- Status register - combinational decoder
    ---------------------------------------------------------------------------
    status_comb(IDLE_IND) <= '1' when (is_bus_off = '1') else
                             '1' when (is_idle = '1') else
                             '0';

    ewl_padded <= '0' & control_registers_out.ewl(7 downto 0);

    status_comb(EWL_IND) <=
        '1' when unsigned(ewl_padded) <= 
                 unsigned(stat_bus(STAT_TX_COUNTER_HIGH downto STAT_TX_COUNTER_LOW))
            else
        '1' when unsigned(ewl_padded) <= 
                 unsigned(stat_bus(STAT_RX_COUNTER_HIGH downto STAT_RX_COUNTER_LOW))
            else
        '0';

    status_comb(TXS_IND) <= is_transmitter;
    status_comb(RXS_IND) <= is_receiver;

    -- Take into account only present number of TXT Buffers!
    txnf_calc_proc : process(txtb_state)
        variable txtb_state_ety : std_logic_vector(G_TXT_BUFFER_COUNT - 1 downto 0);
    begin
        txtb_state_ety := (OTHERS => '0');
        for i in 0 to G_TXT_BUFFER_COUNT - 1 loop
            if (txtb_state(i) = TXT_ETY) then
                txtb_state_ety(i) := '1';
            end if;
        end loop;
        status_comb(TXNF_IND) <= or_reduce(txtb_state_ety);
    end process;

    -- When at least one message is availiable in the buffer
    status_comb(RXNE_IND) <= not rx_empty;

    status_comb(DOR_IND) <= rx_data_overrun;

    status_comb(EFT_IND)  <= stat_bus(STAT_PC_IS_ERR_INDEX);

    status_comb(PEXS_IND) <= stat_bus(STAT_PEXS_INDEX);

    status_comb(RXPE_IND) <= rx_parity_error;

    status_comb(TXPE_IND) <= tx_parity_error;

    status_comb(TXDPE_IND) <= tx_double_parity_error;

    -- TXT Buffer parity error and double parity error
    txpe_flag_proc : process(res_n, clk_sys)
    begin
        if (res_n = '0') then
            tx_parity_error <= '0';
            tx_double_parity_error <= '0';
        elsif rising_edge(clk_sys) then
            for i in 0 to G_TXT_BUFFER_COUNT - 1 loop
                if (txtb_parity_error_valid(i) = '1') then
                    tx_parity_error <= '1';
                end if;

                if (txtb_bb_parity_error(i) = '1') then
                    tx_double_parity_error <= '1';
                end if;
            end loop;

            if (align_wrd_to_reg(control_registers_out.command, CTXPE_IND) = '1') then
                tx_parity_error <= '0';
            end if;

            if (align_wrd_to_reg(control_registers_out.command, CTXDPE_IND) = '1') then
                tx_double_parity_error <= '0';
            end if;
        end if;
    end process;

    traffic_ctrs_gen_true : if G_SUP_TRAFFIC_CTRS generate
        status_comb(STCNT_IND) <= '1';
    end generate traffic_ctrs_gen_true;
    
    traffic_ctrs_gen_false : if not G_SUP_TRAFFIC_CTRS generate
        status_comb(STCNT_IND) <= '0';
    end generate traffic_ctrs_gen_false;
    
    status_comb(31 downto 18) <= (others => '0');
    status_comb(15 downto 12) <= (others => '0');

    ----------------------------------------------------------------------------
    ----------------------------------------------------------------------------
    -- Control registers - Write Data to Driving Bus connection
    ----------------------------------------------------------------------------
    ----------------------------------------------------------------------------

    ---------------------------------------------------------------------------
    -- MODE Register
    ---------------------------------------------------------------------------

    -- FDE - Flexible Data-Rate enabled
    drv_bus(DRV_CAN_FD_ENA_INDEX) <= align_wrd_to_reg(
        control_registers_out.mode, FDE_IND);

    -- Bus monitoring mode
    drv_bus(DRV_BUS_MON_ENA_INDEX) <= align_wrd_to_reg(
        control_registers_out.mode, BMM_IND);

    -- STM - Self test mode 
    drv_bus(DRV_SELF_TEST_ENA_INDEX) <= align_wrd_to_reg(
        control_registers_out.mode, STM_IND);

    -- AFM - Acceptance Filters mode
    drv_bus(DRV_FILTERS_ENA_INDEX) <= align_wrd_to_reg(
        control_registers_out.mode, AFM_IND);

    -- ACF - Acknowledge forbidden mode
    drv_bus(DRV_ACK_FORB_INDEX) <= align_wrd_to_reg(
        control_registers_out.mode, ACF_IND);

    -- ROM - Restricted operation mode
    drv_bus(DRV_ROM_ENA_INDEX) <= align_wrd_to_reg(
        control_registers_out.mode, ROM_IND);
        
    -- TTTM - Time Triggered transmission mode
    drv_bus(DRV_TTTM_ENA_INDEX) <= align_wrd_to_reg(
        control_registers_out.mode, TTTM_IND);

    -- TXBBM - TXT Buffer Backup mode
    drv_txbbm_ena <= align_wrd_to_reg(control_registers_out.mode, TXBBM_IND);
    drv_bus(DRV_TXBBM_ENA_INDEX) <= drv_txbbm_ena;

    -- TSTM - Test mode
    drv_bus(DRV_TSTM_ENA_INDEX) <= align_wrd_to_reg(
        control_registers_out.mode, TSTM_IND);

    ---------------------------------------------------------------------------
    -- COMMAND Register
    ---------------------------------------------------------------------------

    -- CDO - Clear data overrun Flag    
    drv_bus(DRV_CLR_OVR_INDEX) <= align_wrd_to_reg(
        control_registers_out.command, CDO_IND);
    
    -- RRB - Release Receive Buffer
    drv_bus(DRV_ERASE_RX_INDEX) <= align_wrd_to_reg(
        control_registers_out.command, RRB_IND);

    -- ERCRST - Error counter reset
    drv_bus(DRV_ERR_CTR_CLR) <= align_wrd_to_reg(
        control_registers_out.command, ERCRST_IND);

    -- RXFCRST - RX Frame counter reset
    drv_bus(DRV_CLR_RX_CTR_INDEX) <= align_wrd_to_reg(
        control_registers_out.command, RXFCRST_IND);

    -- TXFCRST - TX Frame counter reset
    drv_bus(DRV_CLR_TX_CTR_INDEX) <= align_wrd_to_reg(
        control_registers_out.command, TXFCRST_IND);

    -- CPEXS - Clear protocol exception status (flag)
    drv_bus(DRV_PEXS_CLR_INDEX) <= align_wrd_to_reg(
        control_registers_out.command, CPEXS_IND);

    -- CRXPE - Clear RX Buffer parity error
    drv_bus(DRV_CLR_RXPE_INDEX) <= align_wrd_to_reg(
        control_registers_out.command, CRXPE_IND);

    ---------------------------------------------------------------------------
    -- SETTINGS Register
    ---------------------------------------------------------------------------

    -- RETR_LIM_ENA - Rettransmitt limit enable
    drv_bus(DRV_RETR_LIM_ENA_INDEX) <= align_wrd_to_reg(
        control_registers_out.settings, RTRLE_IND);

    -- RETR_TH - Rettransmitt limit threshold
    drv_bus(DRV_RETR_TH_HIGH downto DRV_RETR_TH_LOW) <= align_wrd_to_reg(
        control_registers_out.settings, RTRTH_H, RTRTH_L);

    -- ENA - CTU CAN FD Core enabled
    drv_bus(DRV_ENA_INDEX) <= align_wrd_to_reg(
        control_registers_out.settings, ENA_IND);
    
    -- NISOFD - Non - ISO FD Flag
    drv_bus(DRV_FD_TYPE_INDEX) <= align_wrd_to_reg(
        control_registers_out.settings, NISOFD_IND);

    -- INT_LOOPBACK - Acknowledge forbidden mode
    drv_bus(DRV_INT_LOOBACK_ENA_INDEX) <= align_wrd_to_reg(
        control_registers_out.settings, ILBP_IND);

    -- PEX - Protocol exception mode
    drv_bus(DRV_PEX_INDEX) <= align_wrd_to_reg(
        control_registers_out.settings, PEX_IND);
        
    -- TBFBO - TXT Buffer Failed Bus off
    txt_buf_failed_bof <= align_wrd_to_reg(
        control_registers_out.settings, TBFBO_IND);
        
    -- Frame filters - drop remote frames
    drv_bus(DRV_FILTER_DROP_RF_INDEX) <= align_wrd_to_reg(
        control_registers_out.settings, FDRF_IND);

    -- Parity check enable
    drv_bus(DRV_PCHK_ENA_INDEX) <= align_wrd_to_reg(
        control_registers_out.settings, PCHKE_IND);

    ---------------------------------------------------------------------------
    -- INT_STAT - Clearing interrupt vector by write
    ---------------------------------------------------------------------------

    -- Set of all Interrupt clears at the same time. We assume that vectors
    -- are addressed at LSB bits!
    drv_bus(DRV_INT_CLR_HIGH downto DRV_INT_CLR_LOW) <= align_wrd_to_reg(
        control_registers_out.int_stat, G_INT_COUNT - 1, 0);

    ---------------------------------------------------------------------------
    -- INT_ENA_SET - Interrupt enable set
    ---------------------------------------------------------------------------
    drv_bus(DRV_INT_ENA_SET_HIGH downto DRV_INT_ENA_SET_LOW) <= align_wrd_to_reg(
            control_registers_out.int_ena_set, G_INT_COUNT - 1, 0);


    ---------------------------------------------------------------------------
    -- INT_ENA_CLR - Interrupt enable clear
    ---------------------------------------------------------------------------
    drv_bus(DRV_INT_ENA_CLR_HIGH downto DRV_INT_ENA_CLR_LOW) <= align_wrd_to_reg(
            control_registers_out.int_ena_clr, G_INT_COUNT - 1, 0);

     
    ---------------------------------------------------------------------------
    -- INT_MASK_SET - Interrupt mask set
    ---------------------------------------------------------------------------
    drv_bus(DRV_INT_MASK_SET_HIGH downto DRV_INT_MASK_SET_LOW) <= align_wrd_to_reg(
            control_registers_out.int_mask_set, G_INT_COUNT - 1, 0);


    ---------------------------------------------------------------------------
    -- INT_MASK_CLR - Interrupt mask clear
    ---------------------------------------------------------------------------
    drv_bus(DRV_INT_MASK_CLR_HIGH downto DRV_INT_MASK_CLR_LOW) <= align_wrd_to_reg(
            control_registers_out.int_mask_clr, G_INT_COUNT - 1, 0);


    ---------------------------------------------------------------------------
    -- BTR - Bit Timing register, Nominal Bit-rate
    ---------------------------------------------------------------------------

    -- TQ_NBT - Time Quanta, Nominal Bit Time
    drv_bus(DRV_TQ_NBT_HIGH downto DRV_TQ_NBT_LOW) <= align_wrd_to_reg(
            control_registers_out.btr, BRP_H, BRP_L);

    -- PRS_NBT - Propagation segment, Nominal Bit Time
    drv_bus(DRV_PRS_NBT_HIGH downto DRV_PRS_NBT_LOW) <= align_wrd_to_reg(
            control_registers_out.btr, PROP_H, PROP_L);    

    -- PH1_NBT - Phase 1, Nominal Bit Time
    drv_bus(DRV_PH1_NBT_HIGH downto DRV_PH1_NBT_LOW) <= align_wrd_to_reg(
            control_registers_out.btr, PH1_H, PH1_L);

    -- PH2_NBT - Phase 2, Nominal Bit Time
    drv_bus(DRV_PH2_NBT_HIGH downto DRV_PH2_NBT_LOW) <= align_wrd_to_reg(
            control_registers_out.btr, PH2_H, PH2_L); 

    -- SJW_NBT - Synchronisation Jump Width, Nominal Bit Time
    drv_bus(DRV_SJW_HIGH downto DRV_SJW_LOW) <= align_wrd_to_reg(
            control_registers_out.btr, SJW_H, SJW_L); 


    ---------------------------------------------------------------------------
    -- BTR FD - Bit Timing register, Data Bit-rate
    ---------------------------------------------------------------------------

    -- TQ_NBT - Time Quanta, Nominal Bit Time
    drv_bus(DRV_TQ_DBT_HIGH downto DRV_TQ_DBT_LOW) <= align_wrd_to_reg(
            control_registers_out.btr_fd, BRP_FD_H, BRP_FD_L);

    -- PRS_NBT - Propagation segment, Nominal Bit Time
    drv_bus(DRV_PRS_DBT_HIGH downto DRV_PRS_DBT_LOW) <= align_wrd_to_reg(
            control_registers_out.btr_fd, PROP_FD_H, PROP_FD_L);    

    -- PH1_NBT - Phase 1, Nominal Bit Time
    drv_bus(DRV_PH1_DBT_HIGH downto DRV_PH1_DBT_LOW) <= align_wrd_to_reg(
            control_registers_out.btr_fd, PH1_FD_H, PH1_FD_L);

    -- PH2_NBT - Phase 2, Nominal Bit Time
    drv_bus(DRV_PH2_DBT_HIGH downto DRV_PH2_DBT_LOW) <= align_wrd_to_reg(
            control_registers_out.btr_fd, PH2_FD_H, PH2_FD_L); 

    -- SJW_NBT - Synchronisation Jump Width, Nominal Bit Time
    drv_bus(DRV_SJW_DBT_HIGH downto DRV_SJW_DBT_LOW) <= align_wrd_to_reg(
            control_registers_out.btr_fd, SJW_FD_H, SJW_FD_L); 


    ---------------------------------------------------------------------------
    -- EWL - Error warning limit
    ---------------------------------------------------------------------------
    drv_bus(DRV_EWL_HIGH downto DRV_EWL_LOW) <= align_wrd_to_reg(
            control_registers_out.ewl, EW_LIMIT_H, EW_LIMIT_L); 


    ---------------------------------------------------------------------------
    -- ERP - Error passive threshold
    ---------------------------------------------------------------------------
    drv_bus(DRV_ERP_HIGH downto DRV_ERP_LOW) <= align_wrd_to_reg(
            control_registers_out.erp, ERP_LIMIT_H, ERP_LIMIT_L); 


    ---------------------------------------------------------------------------
    -- CTR_PRES - Counter preset
    ---------------------------------------------------------------------------

    -- Counter preset value
    drv_bus(DRV_CTR_VAL_HIGH downto DRV_CTR_VAL_LOW) <= align_wrd_to_reg(
            control_registers_out.ctr_pres, CTPV_H, CTPV_L); 
    
    -- Counter preset mask - must be registered, since value is also registered!
    -- This allows setting both by a single access!
    ctr_pres_sel_reg_proc : process(res_n, clk_sys)
    begin
        if (res_n = '0') then
            ctr_pres_sel_q <= (OTHERS => '0');
        elsif rising_edge(clk_sys) then
            ctr_pres_sel_q <= align_wrd_to_reg(control_registers_out.ctr_pres,
                                               EFD_IND, PTX_IND);
        end if;
    end process;
    
    drv_bus(DRV_CTR_SEL_HIGH downto DRV_CTR_SEL_LOW) <= ctr_pres_sel_q;


    ---------------------------------------------------------------------------
    -- FILTER_A_MASK 
    ---------------------------------------------------------------------------    
    drv_bus(DRV_FILTER_A_MASK_HIGH downto DRV_FILTER_A_MASK_LOW) <= align_wrd_to_reg(
            control_registers_out.filter_a_mask, BIT_MASK_A_VAL_H, BIT_MASK_A_VAL_L); 

   
    ---------------------------------------------------------------------------
    -- FILTER_A_VAL
    ---------------------------------------------------------------------------
    drv_bus(DRV_FILTER_A_BITS_HIGH downto DRV_FILTER_A_BITS_LOW) <= align_wrd_to_reg(
            control_registers_out.filter_a_val, BIT_VAL_A_VAL_H, BIT_VAL_A_VAL_L);


    ---------------------------------------------------------------------------
    -- FILTER_B_MASK 
    ---------------------------------------------------------------------------    
    drv_bus(DRV_FILTER_B_MASK_HIGH downto DRV_FILTER_B_MASK_LOW) <= align_wrd_to_reg(
            control_registers_out.filter_b_mask, BIT_MASK_B_VAL_H, BIT_MASK_B_VAL_L); 

   
    ---------------------------------------------------------------------------
    -- FILTER_B_VAL
    ---------------------------------------------------------------------------
    drv_bus(DRV_FILTER_B_BITS_HIGH downto DRV_FILTER_B_BITS_LOW) <= align_wrd_to_reg(
            control_registers_out.filter_b_val, BIT_VAL_B_VAL_H, BIT_VAL_B_VAL_L);


    ---------------------------------------------------------------------------
    -- FILTER_C_MASK 
    ---------------------------------------------------------------------------    
    drv_bus(DRV_FILTER_C_MASK_HIGH downto DRV_FILTER_C_MASK_LOW) <= align_wrd_to_reg(
            control_registers_out.filter_c_mask, BIT_MASK_C_VAL_H, BIT_MASK_C_VAL_L); 

  
    ---------------------------------------------------------------------------
    -- FILTER_C_VAL
    ---------------------------------------------------------------------------
    drv_bus(DRV_FILTER_C_BITS_HIGH downto DRV_FILTER_C_BITS_LOW) <= align_wrd_to_reg(
            control_registers_out.filter_c_val, BIT_VAL_C_VAL_H, BIT_VAL_C_VAL_L);


    ---------------------------------------------------------------------------
    -- FILTER_RAN_LOW
    ---------------------------------------------------------------------------
    drv_bus(DRV_FILTER_RAN_LO_TH_HIGH downto DRV_FILTER_RAN_LO_TH_LOW) <= align_wrd_to_reg(
            control_registers_out.filter_ran_low, BIT_RAN_LOW_VAL_H, BIT_RAN_LOW_VAL_L);


    ---------------------------------------------------------------------------
    -- FILTER_RAN_HIGH
    ---------------------------------------------------------------------------
    drv_bus(DRV_FILTER_RAN_HI_TH_HIGH downto DRV_FILTER_RAN_HI_TH_LOW) <= align_wrd_to_reg(
            control_registers_out.filter_ran_high, BIT_RAN_HIGH_VAL_H, BIT_RAN_HIGH_VAL_L);


    --------------------------------------------------------------------------
    -- FILTER_CONTROL
    ---------------------------------------------------------------------------

    -- Filter A Control
    drv_bus(DRV_FILTER_A_CTRL_HIGH downto DRV_FILTER_A_CTRL_LOW) <= align_wrd_to_reg(
            control_registers_out.filter_control, FAFE_IND, FANB_IND);

    -- Filter B Control
    drv_bus(DRV_FILTER_B_CTRL_HIGH downto DRV_FILTER_B_CTRL_LOW) <= align_wrd_to_reg(
            control_registers_out.filter_control, FBFE_IND, FBNB_IND);

    -- Filter C Control
    drv_bus(DRV_FILTER_C_CTRL_HIGH downto DRV_FILTER_C_CTRL_LOW) <= align_wrd_to_reg(
            control_registers_out.filter_control, FCFE_IND, FCNB_IND);

    -- Filter Range Control
    drv_bus(DRV_FILTER_RAN_CTRL_HIGH downto DRV_FILTER_RAN_CTRL_LOW) <= align_wrd_to_reg(
            control_registers_out.filter_control, FRFE_IND, FRNB_IND);


    --------------------------------------------------------------------------
    -- RX_SETTINGS
    ---------------------------------------------------------------------------

    -- RTSOP - Receive Timestamp options
    drv_bus(DRV_RTSOPT_INDEX) <= align_wrd_to_reg(
        control_registers_out.rx_settings, RTSOP_IND);


    --------------------------------------------------------------------------
    -- RX_DATA
    ---------------------------------------------------------------------------
    -- Signal increment of RX buffer pointer when:
    --  1. Automated mode - we read from RX_DATA register
    --  2. Manual mode - we issue COMMAND[RXRPMV].
    rx_buf_mode <= align_wrd_to_reg(control_registers_out.mode, RXBAM_IND);
    rx_move_cmd <= align_wrd_to_reg(control_registers_out.command, RXRPMV_IND);
    
    drv_bus(DRV_READ_START_INDEX) <=
        control_registers_out.rx_data_read when (rx_buf_mode = RXBAM_ENABLED)
                                           else
                               rx_move_cmd;

    --------------------------------------------------------------------------
    -- TX_COMMAND
    ---------------------------------------------------------------------------
    
    ---------------------------------------------------------------------------
    -- TX_COMMAND register is pipelined on purpose to meet timing. This does
    -- not create a problem, only introduces one clock cycle latency on command
    -- processing!
    ---------------------------------------------------------------------------
    
    tx_cmd_reg_proc : process(clk_sys, soft_res_q_n)
    begin
        if (soft_res_q_n = '0') then
            txtb_sw_cmd.set_rdy <= '0';
            txtb_sw_cmd.set_ety <= '0';
            txtb_sw_cmd.set_abt <= '0';
        elsif (rising_edge(clk_sys)) then
            -- TX SW CMD - Set ready
            txtb_sw_cmd.set_rdy <= align_wrd_to_reg(
                control_registers_out.tx_command, TXCR_IND);
                
            -- TX SW CMD - Set empty
            txtb_sw_cmd.set_ety <= align_wrd_to_reg(
                control_registers_out.tx_command, TXCE_IND);
                
            -- TX SW CMD - Set abort
            txtb_sw_cmd.set_abt <= align_wrd_to_reg(
                control_registers_out.tx_command, TXCA_IND);
        end if;
    end process;
    
    -- TX Command index is regular register, it is therefore pipeline
    -- internally in generated register block!
    
    txtb_cmd_index_gen : for i in 0 to G_TXT_BUFFER_COUNT - 1 generate
    begin
        -- Original TXT Buffers
        txtb_priority_even_gen : if ((i mod 2) = 0) generate
            txtb_sw_cmd_index(i)  <= align_wrd_to_reg(
                control_registers_out.tx_command, TXB1_IND + i);
        end generate;

        -- Backup TXT Buffers
        txtb_priority_odd_gen : if ((i mod 2) = 1) generate
            txtb_sw_cmd_index(i)  <= 
                align_wrd_to_reg(control_registers_out.tx_command, TXB1_IND + i) when (drv_txbbm_ena = '0')
                                                                                 else
                align_wrd_to_reg(control_registers_out.tx_command, TXB1_IND + i - 1);
        end generate;
        
    end generate;

    ---------------------------------------------------------------------------
    -- TX_PRIORITY
    ---------------------------------------------------------------------------

    -- TXT Buffer 1 priority
    -- Dont use register map generated constant to make assigment simple.
    -- It is unlikely position of individual priorities will ever change! 
    txtb_prio_gen : for i in 0 to G_TXT_BUFFER_COUNT - 1 generate
        constant src_index_l : natural := i * 4;
        constant src_index_h : natural := src_index_l + 2;
    begin
        txtb_prorities(i) <= align_wrd_to_reg(
            control_registers_out.tx_priority, src_index_h, src_index_l);
    end generate txtb_prio_gen;

    ---------------------------------------------------------------------------
    -- SSP_CFG
    ---------------------------------------------------------------------------
    
    -- SSP_OFFSET
    drv_bus(DRV_SSP_OFFSET_HIGH downto DRV_SSP_OFFSET_LOW) <= align_wrd_to_reg(
            control_registers_out.ssp_cfg, SSP_OFFSET_H, SSP_OFFSET_L);

    -- SSP_SRC (SSP_DELAY_SELECT)
    drv_bus(DRV_SSP_DELAY_SELECT_HIGH downto DRV_SSP_DELAY_SELECT_LOW) <= align_wrd_to_reg(
            control_registers_out.ssp_cfg, SSP_SRC_H, SSP_SRC_L);


    ---------------------------------------------------------------------------
    ---------------------------------------------------------------------------
    -- Control registers - Read Data to Status Bus connection
    ---------------------------------------------------------------------------
    ---------------------------------------------------------------------------

    ---------------------------------------------------------------------------
    -- DEVICE_ID register
    ---------------------------------------------------------------------------
    Control_registers_in.device_id <= G_DEVICE_ID;

    
    ---------------------------------------------------------------------------
    -- VERSION register
    ---------------------------------------------------------------------------
    version_reg_block : block
        constant length : natural := Control_registers_in.version'length;
    begin
        -- Version minor
        Control_registers_in.version(
            align_reg_to_wrd(VER_MINOR_H, length) downto
            align_reg_to_wrd(VER_MINOR_L, length)) <=
            G_VERSION_MINOR;

        -- Version major
        Control_registers_in.version(
            align_reg_to_wrd(VER_MAJOR_H, length) downto
            align_reg_to_wrd(VER_MAJOR_L, length)) <=
            G_VERSION_MAJOR;
    end block version_reg_block;


    ---------------------------------------------------------------------------
    -- STATUS register
    ---------------------------------------------------------------------------
    Control_registers_in.status <= status_comb;


    ---------------------------------------------------------------------------
    -- INT_STAT register - reading interrupt vector
    ---------------------------------------------------------------------------
    Control_registers_in.int_stat <= INT_PADDING & int_vector;


    ---------------------------------------------------------------------------
    -- INT_ENA_SET register - reading interrupt enable
    ---------------------------------------------------------------------------
    Control_registers_in.int_ena_set <= INT_PADDING & int_ena;


    ---------------------------------------------------------------------------
    -- INT_MASK_SET register - reading interrupt mask
    ---------------------------------------------------------------------------
    Control_registers_in.int_mask_set <= INT_PADDING & int_mask;


    ---------------------------------------------------------------------------
    -- FAULT_STATE register - 
    ---------------------------------------------------------------------------
    fault_state_reg_block : block
        constant length : natural := Control_registers_in.fault_state'length;
    begin
        -- ERA field - Error active
        Control_registers_in.fault_state(align_reg_to_wrd(ERA_IND, length)) <=
            is_err_active;

        -- ERP field - Error passive
        Control_registers_in.fault_state(align_reg_to_wrd(ERP_IND, length)) <=
            is_err_passive;

        -- BOF field - Bus off
        Control_registers_in.fault_state(align_reg_to_wrd(BOF_IND, length)) <=
            is_bus_off;

        -- Pad rest by zeroes
        Control_registers_in.fault_state(
            Control_registers_in.fault_state'length - 1 downto 3) <=
            (OTHERS => '0');
    end block fault_state_reg_block;


    ---------------------------------------------------------------------------
    -- REC Register - Receive error counter
    ---------------------------------------------------------------------------
    rec_reg_block : block
        constant length : natural := Control_registers_in.rec'length;
    begin
        Control_registers_in.rec(
            align_reg_to_wrd(REC_VAL_H, length) downto
            align_reg_to_wrd(REC_VAL_L, length)) <=
            stat_bus(STAT_RX_COUNTER_HIGH downto STAT_RX_COUNTER_LOW);
        Control_registers_in.rec(15 downto 9) <= (OTHERS => '0');
    end block rec_reg_block;


    ---------------------------------------------------------------------------
    -- TEC Register - Transmitt error counter
    ---------------------------------------------------------------------------
    tec_reg_block : block
        constant length : natural := Control_registers_in.tec'length;
    begin
        Control_registers_in.tec(
            align_reg_to_wrd(TEC_VAL_H, length) downto
            align_reg_to_wrd(TEC_VAL_L, length)) <= 
            stat_bus(STAT_TX_COUNTER_HIGH downto STAT_TX_COUNTER_LOW);
            Control_registers_in.tec(15 downto 9) <= (OTHERS => '0');
    end block tec_reg_block;

    ---------------------------------------------------------------------------
    -- ERR_NORM - Error counter Nominal Bit-Rate
    ---------------------------------------------------------------------------
    err_norm_block : block
        constant length : natural := Control_registers_in.err_norm'length;
    begin
        Control_registers_in.err_norm(
            align_reg_to_wrd(ERR_NORM_VAL_H, length) downto
            align_reg_to_wrd(ERR_NORM_VAL_L, length)) <= 
            stat_bus(STAT_ERR_COUNTER_NORM_HIGH downto STAT_ERR_COUNTER_NORM_LOW);
    end block err_norm_block;


    ---------------------------------------------------------------------------
    -- ERR_FD - Error counter Nominal Data-Rate
    ---------------------------------------------------------------------------
    err_fd_block : block
        constant length : natural := Control_registers_in.err_fd'length;
    begin
        Control_registers_in.err_fd(
            align_reg_to_wrd(ERR_FD_VAL_H, length) downto
            align_reg_to_wrd(ERR_FD_VAL_L, length)) <= 
            stat_bus(STAT_ERR_COUNTER_FD_HIGH downto STAT_ERR_COUNTER_FD_LOW);
    end block err_fd_block;


    ---------------------------------------------------------------------------
    -- FILTER_STATUS
    ---------------------------------------------------------------------------
    filter_status_block : block
        constant length : natural := Control_registers_in.filter_status'length;
    begin
    
        -- SFA - Support Filter A -> yes
        sup_filt_A_gen : if (G_SUP_FILTA) generate
            Control_registers_in.filter_status(
                align_reg_to_wrd(SFA_IND, length)) <= '1';
        end generate sup_filt_A_gen;

        -- SFA - Support filter A -> no
        not_sup_filt_A_gen : if (not G_SUP_FILTA) generate
            Control_registers_in.filter_status(
                align_reg_to_wrd(SFA_IND, length)) <= '0';
        end generate not_sup_filt_A_gen;

        -- SFB - Support Filter B -> yes
        sup_filt_B_gen : if (G_SUP_FILTB) generate
            Control_registers_in.filter_status(
                align_reg_to_wrd(SFB_IND, length)) <= '1';
        end generate sup_filt_B_gen;

        -- SFB - Support filter B -> no
        not_sup_filt_B_gen : if (not G_SUP_FILTB) generate
            Control_registers_in.filter_status(
                align_reg_to_wrd(SFB_IND, length)) <= '0';
        end generate not_sup_filt_B_gen;

        -- SFC - Support Filter C -> yes
        sup_filt_C_gen : if (G_SUP_FILTC) generate
            Control_registers_in.filter_status(
                align_reg_to_wrd(SFC_IND, length)) <= '1';
        end generate sup_filt_C_gen;

        -- SFC - Support filter C -> no
        not_sup_filt_C_gen : if (not G_SUP_FILTC) generate
            Control_registers_in.filter_status(
                align_reg_to_wrd(SFC_IND, length)) <= '0';
        end generate not_sup_filt_C_gen;

        -- SFR - Support Filter Range -> yes
        sup_filt_range_gen : if (G_SUP_RANGE) generate
            Control_registers_in.filter_status(
                align_reg_to_wrd(SFR_IND, length)) <= '1';
        end generate sup_filt_range_gen;

        -- SFR - Support filter Range -> no
        not_sup_filt_range_gen : if (not G_SUP_RANGE) generate
            Control_registers_in.filter_status(
                align_reg_to_wrd(SFR_IND, length)) <= '0';
        end generate not_sup_filt_range_gen;

        -- Pad rest by zeroes
        Control_registers_in.filter_status(
            Control_registers_in.filter_status'length - 1 downto 4) <=
            (OTHERS => '0');

    end block filter_status_block;


    ---------------------------------------------------------------------------
    -- RX_MEM_INFO register
    ---------------------------------------------------------------------------
    rx_mem_info_block : block
        constant length : natural := Control_registers_in.rx_mem_info'length;
    begin

        -- RX_BUFF_SIZE field
        Control_registers_in.rx_mem_info(
            align_reg_to_wrd(RX_BUFF_SIZE_H, length) downto
            align_reg_to_wrd(RX_BUFF_SIZE_L, length)) <=
            rx_buf_size;

        -- RX_MEM_FREE field
        Control_registers_in.rx_mem_info(
            align_reg_to_wrd(RX_MEM_FREE_H, length) downto
            align_reg_to_wrd(RX_MEM_FREE_L, length)) <=
            rx_mem_free;

        -- Padd rest by zeroes
        Control_registers_in.rx_mem_info(31 downto 29) <= (OTHERS => '0');
        Control_registers_in.rx_mem_info(15 downto 13) <= (OTHERS => '0');

    end block rx_mem_info_block;


    ---------------------------------------------------------------------------
    -- RX_POINTERS register
    ---------------------------------------------------------------------------
    rx_pointers_block : block
        constant length : natural := Control_registers_in.rx_pointers'length;
    begin

        -- RX_WPP field - RX Write Pointer position
        Control_registers_in.rx_pointers(
            align_reg_to_wrd(RX_WPP_H, length) downto
            align_reg_to_wrd(RX_WPP_L, length)) <=
            rx_write_pointer;

        -- RX_RPP field - RX Read Pointer position
        Control_registers_in.rx_pointers(
            align_reg_to_wrd(RX_RPP_H, length) downto
            align_reg_to_wrd(RX_RPP_L, length)) <=
            rx_read_pointer;

        -- Padd rest by zeroes
        Control_registers_in.rx_pointers(31 downto 28) <= (OTHERS => '0');
        Control_registers_in.rx_pointers(15 downto 12) <= (OTHERS => '0');

    end block rx_pointers_block;


    ---------------------------------------------------------------------------
    -- RX_STATUS register
    ---------------------------------------------------------------------------
    rx_status_block : block
        constant length : natural := Control_registers_in.rx_status'length;
    begin

        -- RXE field - RX Buffer Empty field
        Control_registers_in.rx_status(
            align_reg_to_wrd(RXE_IND, length)) <=
            rx_empty;

        -- RXF field - RX Buffer Full field
        Control_registers_in.rx_status(
            align_reg_to_wrd(RXF_IND, length)) <=
            rx_full;

        -- RXFRC field - RX Buffer Frame count
        Control_registers_in.rx_status(
            align_reg_to_wrd(RXFRC_H, length) downto
            align_reg_to_wrd(RXFRC_L, length)) <=
            rx_frame_count;

        -- RXMOF - RX Buffer middle of frame
        Control_registers_in.rx_status(
            align_reg_to_wrd(RXMOF_IND, length)) <=
            rx_mof;

        -- Padd rest by zeroes
        Control_registers_in.rx_status(15) <= '0';
        Control_registers_in.rx_status(3) <= '0';

    end block rx_status_block;


    ---------------------------------------------------------------------------
    -- RX_DATA register - Read data word from RX Buffer FIFO.
    ---------------------------------------------------------------------------
    Control_registers_in.rx_data <= rx_read_buff;


    ---------------------------------------------------------------------------
    -- TX_STATUS register
    ---------------------------------------------------------------------------
    tx_status_block : block
        constant length : natural := Control_registers_in.tx_status'length;
    begin

        -- TX1S - TXT Buffer 1 status field
        Control_registers_in.tx_status(
            align_reg_to_wrd(TX1S_H, length) downto
            align_reg_to_wrd(TX1S_L, length)) <=
            txtb_state(0);
     
        -- TX2S - TXT Buffer 2 status field
        Control_registers_in.tx_status(
            align_reg_to_wrd(TX2S_H, length) downto
            align_reg_to_wrd(TX2S_L, length)) <=
            txtb_state(1);

        -- TX3S, TX4S, TX5S, TX6S, TX7S, TX8S  - TXT Buffer 3-8 status fields
        -- We dont use generated constants and rely on field positions.
        -- It is unlikely these will change!
        txtb_status_gen_loop : for buf_index in 2 to 7 generate
            constant index_l : natural := buf_index * 4;
            constant index_h : natural := index_l + 3;
        begin
            txtb3_stat_gen_true : if (G_TXT_BUFFER_COUNT > buf_index) generate
                Control_registers_in.tx_status(
                    align_reg_to_wrd(index_h, length) downto
                    align_reg_to_wrd(index_l, length)) <=
                    txtb_state(buf_index);
            end generate txtb3_stat_gen_true;

            txtb3_stat_gen_false : if (G_TXT_BUFFER_COUNT <= buf_index) generate
                Control_registers_in.tx_status(
                    align_reg_to_wrd(index_h, length) downto
                    align_reg_to_wrd(index_l, length)) <= TXT_NOT_EXIST;
            end generate txtb3_stat_gen_false;

        end generate txtb_status_gen_loop;

    end block tx_status_block;

    ---------------------------------------------------------------------------
    -- TXTB_INFO register
    ---------------------------------------------------------------------------
    txtb_info_block : block
        constant length : natural := Control_registers_in.txtb_info'length;
    begin
        Control_registers_in.txtb_info(
            align_reg_to_wrd(TXT_BUFFER_COUNT_H, length) downto
            align_reg_to_wrd(TXT_BUFFER_COUNT_L, length)) <=
            std_logic_vector(to_unsigned(G_TXT_BUFFER_COUNT, 4));
        Control_registers_in.txtb_info(15 DOWNTO 4) <= (OTHERS => '0');
    end block;

    ---------------------------------------------------------------------------
    -- ERR_CAPT register
    ---------------------------------------------------------------------------
    err_capt_block : block
        constant length : natural := Control_registers_in.err_capt'length;
    begin

        -- ERR_POS - Error position field
        Control_registers_in.err_capt(
            align_reg_to_wrd(ERR_POS_H, length) downto
            align_reg_to_wrd(ERR_POS_L, length)) <=
            stat_bus(STAT_ERC_ERR_POS_HIGH downto STAT_ERC_ERR_POS_LOW);

        -- ERR_TYPE - Error type field
        Control_registers_in.err_capt(
            align_reg_to_wrd(ERR_TYPE_H, length) downto
            align_reg_to_wrd(ERR_TYPE_L, length)) <=
            stat_bus(STAT_ERC_ERR_TYPE_HIGH downto STAT_ERC_ERR_TYPE_LOW);

    end block err_capt_block;

    ---------------------------------------------------------------------------
    -- RETR_CTR register
    ---------------------------------------------------------------------------
    retr_ctr_block : block
        constant length : natural := Control_registers_in.retr_ctr'length;
    begin

        -- ERR_POS - Error position field
        Control_registers_in.retr_ctr(
            align_reg_to_wrd(RETR_CTR_VAL_H, length) downto
            align_reg_to_wrd(RETR_CTR_VAL_L, length)) <=
            stat_bus(STAT_RETR_CTR_HIGH downto STAT_RETR_CTR_LOW);
        Control_registers_in.retr_ctr(7 downto 4) <= (OTHERS => '0');

    end block retr_ctr_block;

    ---------------------------------------------------------------------------
    -- ALC register
    ---------------------------------------------------------------------------
    alc_block : block
        constant length : natural := Control_registers_in.alc'length;
    begin
    
        -- ALC_ID_FIELD - Arbitration lost capture ID field
        Control_registers_in.alc(
            align_reg_to_wrd(ALC_ID_FIELD_H, length) downto
            align_reg_to_wrd(ALC_ID_FIELD_L, length)) <=
            stat_bus(STAT_ALC_ID_FIELD_HIGH downto STAT_ALC_ID_FIELD_LOW);

        -- ALC_ID_BIT - Arbitration lost capture bit position
        Control_registers_in.alc(
            align_reg_to_wrd(ALC_BIT_H, length) downto
            align_reg_to_wrd(ALC_BIT_L, length)) <=
            stat_bus(STAT_ALC_BIT_HIGH downto STAT_ALC_BIT_LOW);

    end block alc_block;

    ---------------------------------------------------------------------------
    -- TS_INFO register
    ---------------------------------------------------------------------------
    ts_info_block : block
        constant length : natural := Control_registers_in.ts_info'length;
    begin

        -- TS_BITS - Number of active timestamp bits
        Control_registers_in.ts_info(
            align_reg_to_wrd(TS_BITS_H, length) downto
            align_reg_to_wrd(TS_BITS_L, length)) <=
                std_logic_vector(to_unsigned(G_TS_BITS, 6));

    end block ts_info_block;

    Control_registers_in.ts_info(7 downto 6) <= "00";

    ---------------------------------------------------------------------------
    -- TRV_DELAY register
    ---------------------------------------------------------------------------
    trv_delay_block : block
        constant length : natural := Control_registers_in.trv_delay'length;
    begin
    
        Control_registers_in.trv_delay(
            align_reg_to_wrd(TRV_DELAY_VALUE_H, length) downto
            align_reg_to_wrd(TRV_DELAY_VALUE_L, length)) <=
            trv_delay;
        Control_registers_in.trv_delay(15 downto 7) <= (others => '0');
    end block trv_delay_block;


    ---------------------------------------------------------------------------
    -- RX_FR_CTR register
    ---------------------------------------------------------------------------
    rx_fr_ctr_block : block
        constant length : natural := Control_registers_in.rx_fr_ctr'length;
    begin

        Control_registers_in.rx_fr_ctr(
            align_reg_to_wrd(RX_FR_CTR_VAL_H, length) downto
            align_reg_to_wrd(RX_FR_CTR_VAL_L, length)) <=
            stat_bus(STAT_RX_CTR_HIGH downto STAT_RX_CTR_LOW);

    end block rx_fr_ctr_block;


    ---------------------------------------------------------------------------
    -- TX_FR_CTR register
    ---------------------------------------------------------------------------
    tx_fr_ctr_block : block
        constant length : natural := Control_registers_in.tx_fr_ctr'length;
    begin

        Control_registers_in.tx_fr_ctr(
            align_reg_to_wrd(TX_FR_CTR_VAL_H, length) downto
            align_reg_to_wrd(TX_FR_CTR_VAL_L, length)) <=
            stat_bus(STAT_TX_CTR_HIGH downto STAT_TX_CTR_LOW);

    end block tx_fr_ctr_block;


    ---------------------------------------------------------------------------
    -- DEBUG register
    ---------------------------------------------------------------------------
    debug_register_block : block
        constant length : natural := Control_registers_in.debug_register'length;
    begin

        -- STUFF_COUNT - Counter of stuffed bits modulo 8
        Control_registers_in.debug_register(
            align_reg_to_wrd(STUFF_COUNT_H, length) downto
            align_reg_to_wrd(STUFF_COUNT_L, length)) <=
            stat_bus(STAT_BS_CTR_HIGH downto STAT_BS_CTR_LOW);

        -- DESTUFF_COUNT - Counter of de-stuffed bits modulo 8
        Control_registers_in.debug_register(
            align_reg_to_wrd(DESTUFF_COUNT_H, length) downto
            align_reg_to_wrd(DESTUFF_COUNT_L, length)) <=
            stat_bus(STAT_BD_CTR_HIGH downto STAT_BD_CTR_LOW);

        -- PC_ARB field - Protocol control FSM - Start of frame field
        Control_registers_in.debug_register(
            align_reg_to_wrd(PC_SOF_IND, length)) <=
            stat_bus(STAT_PC_IS_SOF);

        -- PC_ARB field - Protocol control FSM - arbitration field
        Control_registers_in.debug_register(
            align_reg_to_wrd(PC_ARB_IND, length)) <=
            stat_bus(STAT_PC_IS_ARBITRATION_INDEX);

        -- PC_CON field - Protocol control FSM - control field
        Control_registers_in.debug_register(
            align_reg_to_wrd(PC_CON_IND, length)) <=
            stat_bus(STAT_PC_IS_CONTROL_INDEX);

        -- PC_DAT field - Protocol control FSM - data field
        Control_registers_in.debug_register(
            align_reg_to_wrd(PC_DAT_IND, length)) <=
            stat_bus(STAT_PC_IS_DATA_INDEX);

        -- PC_STC field - Protocol control FSM - Stuff count field
        Control_registers_in.debug_register(
            align_reg_to_wrd(PC_STC_IND, length)) <=
            stat_bus(STAT_PC_IS_STUFF_COUNT_INDEX);

        -- PC_CRC field - Protocol control FSM - CRC field
        Control_registers_in.debug_register(
            align_reg_to_wrd(PC_CRC_IND, length)) <=
            stat_bus(STAT_PC_IS_CRC_INDEX);

        -- PC_CRCD field - Protocol control FSM - CRC Delimiter field
        Control_registers_in.debug_register(
            align_reg_to_wrd(PC_CRCD_IND, length)) <=
            stat_bus(STAT_PC_IS_CRC_DELIM_INDEX);
            
        -- PC_ACK field - Protocol control FSM - ACK field
        Control_registers_in.debug_register(
            align_reg_to_wrd(PC_ACK_IND, length)) <=
            stat_bus(STAT_PC_IS_ACK_FIELD_INDEX);

        -- PC_ACKD field - Protocol control FSM - ACK Delimiter field
        Control_registers_in.debug_register(
            align_reg_to_wrd(PC_ACKD_IND, length)) <=
            stat_bus(STAT_PC_IS_ACK_DELIM_INDEX);

        -- PC_EOF field - Protocol control FSM - EOF field
        Control_registers_in.debug_register(
            align_reg_to_wrd(PC_EOF_IND, length)) <=
            stat_bus(STAT_PC_IS_EOF_INDEX);

        -- PC_OVR field - Protocol control FSM - Overload frame field
        Control_registers_in.debug_register(
            align_reg_to_wrd(PC_OVR_IND, length)) <=
            stat_bus(STAT_PC_IS_OVERLOAD_INDEX);

        -- PC_INT field - Protocol control FSM - Intermission frame field
        Control_registers_in.debug_register(
            align_reg_to_wrd(PC_INT_IND, length)) <=
            stat_bus(STAT_PC_IS_INTERMISSION_INDEX);

        -- PC_SUSP field - Protocol control FSM - Suspend transmission frame field
        Control_registers_in.debug_register(
            align_reg_to_wrd(PC_SUSP_IND, length)) <=
            stat_bus(STAT_PC_IS_SUSPEND_INDEX);

        -- Pad rest by zeroes
        Control_registers_in.debug_register(31 downto 19) <= (OTHERS => '0');

    end block debug_register_block;


    ---------------------------------------------------------------------------
    -- YOLO register
    ---------------------------------------------------------------------------
    yolo_register_block : block
        constant length : natural := Control_registers_in.yolo_reg'length;
    begin

        Control_registers_in.yolo_reg(
            align_reg_to_wrd(YOLO_VAL_H, length) downto
            align_reg_to_wrd(YOLO_VAL_L, length)) <=
            YOLO_VAL_RSTVAL;

    end block yolo_register_block;


    ---------------------------------------------------------------------------
    -- TIMESTAMP_LOW, TIMESTAMP_HIGH registers
    ---------------------------------------------------------------------------
    timestamp_registers_block : block
        constant ts_low_l : natural := Control_registers_in.timestamp_low'length;
        constant ts_high_l : natural := Control_registers_in.timestamp_high'length;
    begin

        Control_registers_in.timestamp_low(
            align_reg_to_wrd(TIMESTAMP_LOW_H, ts_low_l) downto
            align_reg_to_wrd(TIMESTAMP_LOW_L, ts_low_l)) <=
            timestamp(31 downto 0);

        Control_registers_in.timestamp_high(
            align_reg_to_wrd(TIMESTAMP_HIGH_H, ts_high_l) downto
            align_reg_to_wrd(TIMESTAMP_HIGH_L, ts_high_l)) <=
            timestamp(63 downto 32);

    end block timestamp_registers_block;

   
    ----------------------------------------------------------------------------
    -- Driving bus assignment
    ----------------------------------------------------------------------------
    -- Note:  All unused signals indices are assigned to zero!
    drv_bus(80 downto 61)   <= (OTHERS => '0');
    drv_bus(349 downto 331) <= (OTHERS => '0');
    drv_bus(365 downto 358) <= (OTHERS => '0');
    drv_bus(371 downto 368) <= (OTHERS => '0');
    drv_bus(399 downto 383) <= (OTHERS => '0');
    drv_bus(459 downto 445) <= (OTHERS => '0');
    drv_bus(464 downto 462) <= (OTHERS => '0');
    drv_bus(609 downto 601) <= (OTHERS => '0');
    drv_bus(579 downto 570) <= (OTHERS => '0');
    drv_bus(519 downto 514) <= (OTHERS => '0');
    drv_bus(506 downto 478) <= (OTHERS => '0');
    drv_bus(444 downto 430) <= (OTHERS => '0');

    drv_bus(1023 downto 876)<= (OTHERS => '0');

    drv_bus(863 downto 844) <= (OTHERS => '0');
    drv_bus(831 downto 812) <= (OTHERS => '0');
    drv_bus(799 downto 780) <= (OTHERS => '0');
    drv_bus(767 downto 748) <= (OTHERS => '0');
    drv_bus(735 downto 614) <= (OTHERS => '0');
    
    drv_bus(613 downto 610) <= (OTHERS => '0');
    drv_bus(600 downto 580) <= (OTHERS => '0');
    drv_bus(569 downto 552) <= (OTHERS => '0');
    drv_bus(551 downto 520) <= (OTHERS => '0');

    drv_bus(372)            <= '0';
    drv_bus(461)            <= '0';
    drv_bus(367)            <= '0';
    drv_bus(366)            <= '0';
    drv_bus(357)            <= '0';
    drv_bus(356)            <= '0';
    drv_bus(355)            <= '0';

    -- <RELEASE_OFF>
    ----------------------------------------------------------------------------
    -- Assertions / Functional coverage
    ----------------------------------------------------------------------------
    
    -- psl default clock is rising_edge(clk_sys);

    -- psl no_simul_two_reg_block_access_asrt : assert never
    --   (control_registers_cs_reg = '1' and test_registers_cs_reg = '1')
    --   report "Control registers and test registers can't be accessed at once!";

    -- psl no_rxpe_when_parity_disabled_cov : assert never
    --  (drv_bus(DRV_PCHK_ENA_INDEX) = '0' and rx_parity_error = '1')
    --  report "RX Parity error generated when SETTINGS[PCHKE] is disabled.";

    -- psl no_txpe_when_parity_disabled_cov : assert never
    --  (drv_bus(DRV_PCHK_ENA_INDEX) = '0' and tx_parity_error = '1')
    --  report "TX Parity error generated when SETTINGS[PCHKE] is disabled.";

    -- psl no_txdpe_when_parity_disabled_cov : assert never
    --  (drv_bus(DRV_PCHK_ENA_INDEX) = '0' and tx_double_parity_error = '1')
    --  report "TX Double parity error generated when SETTINGS[PCHKE] is disabled.";

    txtb_func_cov_gen : for i in 0 to G_TXT_BUFFER_COUNT - 1 generate
    
    -- psl txtb_in_parit_error_when_parity_disabled_asrt : assert never
    --    {drv_bus(DRV_PCHK_ENA_INDEX) = '0' and txtb_state(i) = TXT_PER}
    --    report "TXT Buffer in 'Parity error' state when SETTINGS[PCHKE] is disabled.";
    
    end generate;

    -- psl rx_buf_automatic_mode_cov : cover
    --   {rx_buf_mode = RXBAM_ENABLED};
    
    -- psl rx_buf_manual_mode_cov : cover
    --   {rx_buf_mode = RXBAM_DISABLED};
    
    -- <RELEASE_ON>

end architecture;
