/******************************************************************************
 *
 * ISO16845 Compliance tests
 * Copyright (C) 2021-present Ondrej Ille
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this SW component and associated documentation files (the "Component"),
 * to use, copy, modify, merge, publish, distribute the Component for
 * educational, research, evaluation, self-interest purposes. Using the
 * Component for commercial purposes is forbidden unless previously agreed with
 * Copyright holder.
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Component.
 *
 * THE COMPONENT IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHTHOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE COMPONENT OR THE USE OR OTHER DEALINGS
 * IN THE COMPONENT.
 *
 * @author Ondrej Ille, <ondrej.ille@gmail.com>
 * @date 11.4.2020
 *
 *****************************************************************************/

/******************************************************************************
 *
 * @test ISO16845 7.3.1
 *
 * @brief This test verifies that the IUT tolerates up to 7 consecutive
 *        dominant bits after sending an active error flag.
 * @version Classical CAN, CAN FD Tolerant, CAN FD Enabled
 *
 * Test variables:
 *  Classical CAN, CAN FD Tolerant, CAN FD Enabled
 *      FDF = 0
 *
 *  CAN FD Enabled
 *      FDF = 1
 *
 * Elementary test cases:
 *      #1 lengthening the error flag by 1 dominant bit;
 *      #2 lengthening the error flag by 4 dominant bits;
 *      #3 lengthening the error flag by 7 dominant bits.
 *
 * Setup:
 *  The IUT is left in the default state.
 *
 * Execution:
 *  The LT causes the IUT to generate an error frame in data field.
 *  The LT lengthens the error flag generated by the IUT according to
 *  elementary test cases.
 *
 * Response:
 *  After sending the active error flag, the IUT sends recessive bits.
 *****************************************************************************/

#include <iostream>
#include <unistd.h>
#include <chrono>

#include "TestBase.h"

using namespace can;
using namespace test;

class TestIso_7_3_1 : public test::TestBase
{
    public:

        void ConfigureTest()
        {
            FillTestVariants(VariantMatchType::CommonAndFd);
            for (size_t i = 0; i < 3; i++)
            {
                AddElemTest(TestVariant::Common, ElemTest(i + 1, FrameKind::Can20));
                AddElemTest(TestVariant::CanFdEna, ElemTest(i + 1, FrameKind::CanFd));
            }
        }

        int RunElemTest([[maybe_unused]] const ElemTest &elem_test,
                        [[maybe_unused]] const TestVariant &test_variant)
        {
            frm_flags = std::make_unique<FrameFlags>(elem_test.frame_kind_,
                            RtrFlag::Data);
            gold_frm = std::make_unique<Frame>(*frm_flags, 1, &error_data);
            RandomizeAndPrint(gold_frm.get());

            drv_bit_frm = ConvBitFrame(*gold_frm);
            mon_bit_frm = ConvBitFrame(*gold_frm);

            size_t prolonged_by = (3 * (elem_test.index_ - 1)) + 1;
            TestMessage("Prolonging Active Error flag by: %zu", prolonged_by);

            /**************************************************************************************
             * Modify test frames:
             *   1. Monitor frame as if received.
             *   2. Force 7-th bit of Data frame to opposite, this should be stuff bit!
             *      This will cause stuff error!
             *   3. Insert Active Error frame from 8-th bit of data frame!
             *   4. Prolong Active error flag by 1,4,7 bits respectively.
             *      Prolong Monitored error delimier by 1,4,7 Recessive bits!
             *************************************************************************************/
            mon_bit_frm->ConvRXFrame();

            drv_bit_frm->GetBitOf(6, BitKind::Data)->FlipVal();

            mon_bit_frm->InsertActErrFrm(7, BitKind::Data);
            drv_bit_frm->InsertActErrFrm(7, BitKind::Data);

            size_t num_bits_to_insert;
            if (elem_test.index_ == 1)
                num_bits_to_insert = 1;
            else if (elem_test.index_ == 2)
                num_bits_to_insert = 4;
            else
                num_bits_to_insert = 7;

            /* Prolong driven frame by 1,4,7 DOMINANT bits */
            size_t drv_last_err_flg_index = drv_bit_frm->GetBitIndex(
                drv_bit_frm->GetBitOf(5, BitKind::ActErrFlag));
            for (size_t k = 0; k < num_bits_to_insert; k++)
                drv_bit_frm->InsertBit(BitKind::ActErrFlag, BitVal::Dominant,
                                            drv_last_err_flg_index);

            /* Prolong monitored frame by 1,4,7 RECESSIVE bits */
            size_t mon_last_err_flg_index = mon_bit_frm->GetBitIndex(
                mon_bit_frm->GetBitOf(0, BitKind::ErrDelim));
            for (size_t k = 0; k < num_bits_to_insert; k++)
                mon_bit_frm->InsertBit(BitKind::ErrDelim, BitVal::Recessive,
                                            mon_last_err_flg_index);

            drv_bit_frm->Print(true);
            mon_bit_frm->Print(true);

            /**************************************************************************************
             * Execute test
             *************************************************************************************/
            PushFramesToLT(*drv_bit_frm, *mon_bit_frm);
            RunLT(true, true);
            CheckLTResult();
            CheckNoRxFrame();

            FreeTestObjects();
            return FinishElemTest();
        }
};