/*****************************************************************************
 * vim:tw=78
 *
 * Copyright (C) 2022 Richard Hacker (lerichi at gmx dot net),
 *                    Bjarne von Horn (vh at igh dot de).
 *
 * This file is part of the PdCom library.
 *
 * The PdCom library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 *
 * The PdCom library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with the PdCom library. If not, see <http://www.gnu.org/licenses/>.
 *
 *****************************************************************************/

/** @file */

#ifndef PDCOM5_MESSAGEMANAGERBASE_H
#define PDCOM5_MESSAGEMANAGERBASE_H

#include <chrono>
#include <cstdint>
#include <memory>
#include <pdcom5_export.h>
#include <string>
#include <vector>

namespace PdCom {
namespace impl {
class Process;
}

enum class LogLevel : int {
    Reset     = -1, /**< Message is reset */
    Emergency = 0,  /**< Emergency log level */
    Alert     = 1,  /**< Alert log level */
    Critical  = 2,  /**< Critical log level */
    Error     = 3,  /**< Error log level */
    Warn      = 4,  /**< Warn log level */
    Info      = 6,  /**< Info log level */
    Debug     = 7,  /**< Debug log level */
    Trace     = 8,  /**< Trace log level */
};

/** Message structure */
struct Message
{
    uint32_t seqNo;                /**< sequence number. Note that seqNo
                                    wraps naturally */
    LogLevel level;                /**< severity */
    std::string path;              /**< event's path */
    std::chrono::nanoseconds time; /**< event time in nanoseconds since epoch */
    std::string text;              /**< Text of message */
    int index;                     /**< -1 for scalar;
                                        index in case of a vector */
};

class PDCOM5_PUBLIC MessageManagerBase
{
  public:
    MessageManagerBase();
    MessageManagerBase(MessageManagerBase &&) noexcept;
    MessageManagerBase(MessageManagerBase const &) = delete;
    MessageManagerBase &operator=(MessageManagerBase &&) noexcept;
    MessageManagerBase &operator=(MessageManagerBase const &) = delete;

  protected:
    /** Request specific message from history
     *
     * The message is received in getMessageReply().
     *
     * If the seqNo is not available, getMessageReply()
     * is called with null \p path and requested seqNo.
     *
     * @param seqNo sequence number of message
     */
    void getMessage(uint32_t seqNo) const;

    /** Request a list of all active messages.
     *
     * This will return a list of active process messages as well as
     * the last message in activeMessagesReply() if its state is
     * Reset.
     */
    void activeMessages() const;

    virtual ~MessageManagerBase();

  private:
    friend impl::Process;
    std::weak_ptr<impl::Process> process_;

    /** Message event from process
     *
     * This method is called whenever a message event is generated by
     * the process.
     *
     * Reimplement this method to receive them.
     *
     * @param message
     */
    virtual void processMessage(Message message);

    /** Reply to getMessage()
     *
     * Reimplement this method to receive responses to getMessage().
     *
     * @param message
     */
    virtual void getMessageReply(Message message);

    /** Reply to activeMessages()
     *
     * Reimplement this method to receive the message list.
     *
     * The process messages received in processMessage() has strictly
     * increasing sequence numbers and time.
     *
     * @param messageList
     */
    virtual void activeMessagesReply(std::vector<Message> messageList);
};
}  // namespace PdCom


#endif  // PDCOM5_MESSAGEMANAGERBASE_H
