/*****************************************************************************
 *
 * $Id$
 *
 * Copyright (C) 2009 - 2012  Richard Hacker <lerich@gmx.net>
 *                            Florian Pose <fp@igh-essen.com>
 *
 * This file is part of the PdCom library.
 *
 * The PdCom library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 *
 * The PdCom library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with the PdCom Library. If not, see <http://www.gnu.org/licenses/>.
 *
 ****************************************************************************/

#ifndef PDCOM_VARIABLETIME_H
#define PDCOM_VARIABLETIME_H

#include <sys/time.h>

#include <string>

namespace PdCom {

/****************************************************************************/

/** ProcessVariable time.
 *
 * This is the time class of PdCom. It can contain absolute times (since
 * epoch) and relative times. The base unit is always a second.
 */

class Time {
    public:
        /** Constructor.
         *
         * Sets the time value to zero.
         */
        Time();

        /** Copy constructor.
         */
        Time(const Time &);

        /** Constructor with double parameter.
         */
        Time(double);

        /** Constructor with time_t parameter.
         */
        Time(time_t);

        /** Assignment operator. */
        Time &operator=(double);

        /** Combined addition and assignment operator.
         */
        Time &operator+=(const Time &);

        /** Combined substration and assignment operator.
         */
        Time &operator-=(const Time &);

        /** Addition operator.
         *
         * \return Sum of the time and the parameter.
         */
        Time operator+(const Time &) const;

        /** Substraction operator.
         *
         * \return Difference between the time and the parameter.
         */
        Time operator-(const Time &) const;

        /** Multiplication operator.
         *
         * \return Product of the time and the parameter.
         */
        Time operator*(
                double seconds /**< Time to multiply with. */
                ) const;

        /** Equal operator.
         *
         * \retval True if the time is equal to the parameter.
         */
        bool operator==(const Time &) const;

        /** Greater operator.
         *
         * \retval True if the time is greater than the parameter.
         */
        bool operator>(const Time &) const;

        /** Greater-or-equal operator.
         *
         * \retval True if the time is greater than or equal to the
         *         parameter.
         */
        bool operator>=(const Time &) const;

        /** Less operator.
         *
         * \retval True if the time is less than the parameter.
         */
        bool operator<(const Time &) const;

        /** Less-or-equal operator.
         *
         * \retval True if the time is less than or equal to the
         *         parameter.
         */
        bool operator<=(const Time &) const;

        /** Conversion to bool.
         *
         * \retval True if the time is non-zero.
         */
        operator bool() const;

        /** Conversion to double.
         *
         * \returns Seconds and fractional seconds in a double value.
         */
        operator double() const;

        /** String representation
         *
         * \return String
         */
        std::string str() const;

    private:
        struct timeval t;
};

} // namespace PdCom

#endif // PDCOM_VARIABLETIME_H

/****************************************************************************/
