/*****************************************************************************
 *
 * Testmanager - Graphical Automation and Visualisation Tool
 *
 * Copyright (C) 2023  Daniel Ramirez <dr@igh.de>
 *
 * This file is part of Testmanager.
 *
 * Testmanager is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option)
 * any later version.
 *
 * Testmanager is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with Testmanager. If not, see <http://www.gnu.org/licenses/>.
 *
 ****************************************************************************/

#include "Parameter.h"

#include "ParameterSetWidget.h"

#include <pdcom5/Subscriber.h>
#include <pdcom5/Subscription.h>

#include <QDebug>
#include <QJsonObject>

using Pd::Parameter;

/*****************************************************************************
 * public
 ****************************************************************************/

Parameter::Parameter(Pd::ParameterSetWidget *widget) :
    widget(widget), dataPresent(false), savedValue(0.0), currentValue(0.0)
{}

/****************************************************************************/

Parameter::~Parameter()
{}

/****************************************************************************/

void Parameter::fromJson(const QJsonValue &value)
{
    QJsonObject obj(value.toObject());

    url = obj["url"].toString();
    connectionPath =
        url.adjusted(QUrl::RemoveAuthority | QUrl::RemoveScheme).toString();
    savedValue = obj["value"].toDouble();
}

/****************************************************************************/

void Parameter::replaceUrl(const QUrl &oldUrl, const QUrl &newUrl)
{
    QUrl dataSourceUrl(url.adjusted(
            QUrl::RemovePassword | QUrl::RemovePath | QUrl::RemoveQuery
            | QUrl::RemoveFragment));
    if (dataSourceUrl != oldUrl) {
        return;
    }

    url.setScheme(newUrl.scheme());
    url.setAuthority(newUrl.authority());  // user, pass, host, port
}

/****************************************************************************/

void Parameter::connectParameter(QtPdCom::Process *p)
{
    if (p) {
#ifdef DEBUG_PD_PARAMETER
        qDebug() << __func__ << url;
#endif
        // Subscribe to process variable as ScalarSubscriber
        setVariable((PdCom::Process *) p, connectionPath);
    }
}

/*****************************************************************************
 * private
 ****************************************************************************/

void Parameter::newValues(std::chrono::nanoseconds)
{
    dataPresent = true;
    PdCom::details::copyData(
            &currentValue,
            PdCom::details::TypeInfoTraits<double>::type_info.type, getData(),
            getVariable().getTypeInfo().type, 1);
#ifdef DEBUG_PD_PARAMETER
    qDebug() << __func__ << url << currentValue;
#endif
    emit dataChanged();
}

/****************************************************************************/

void Parameter::stateChange(PdCom::Subscription::State state)
{
#ifdef DEBUG_PD_PARAMETER
    qDebug() << __func__ << url << (int) state;
#endif
    if (state != PdCom::Subscription::State::Active) {
        dataPresent = false;
        emit dataChanged();
    }
}

/****************************************************************************/
