/*****************************************************************************
 *
 * Copyright (C) 2023 Daniel Ramirez <dr@igh.de>
 *
 * This file is part of the QtPdWidgets library.
 *
 * The QtPdWidgets library is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the License,
 * or (at your option) any later version.
 *
 * The QtPdWidgets library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with the QtPdWidgets Library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 ****************************************************************************/

#include "ParameterSetWidget.h"

#include "Parameter.h"

#include <QAction>
#include <QComboBox>
#include <QDir>
#include <QFile>
#include <QFileDialog>
#include <QFileSystemModel>
#include <QFileSystemWatcher>
#include <QJsonArray>
#include <QJsonDocument>
#include <QJsonObject>
#include <QLabel>
#include <QListView>
#include <QMenu>
#include <QPushButton>
#include <QContextMenuEvent>
#include <QHBoxLayout>

using Pd::ParameterSetWidget;

/****************************************************************************/

struct ParameterSetWidget::Impl
{
    Impl(ParameterSetWidget *widget):
        widget(widget),
        layout(widget),
        label(widget),
        comboBox(widget),
        applyButton(widget),
        boxModel(widget)
    {
        layout.addWidget(&label, 0);
        updateProcessPixmap();

        boxModel.setReadOnly(true);
        boxModelView.setModel(&boxModel);
        comboBox.setModel(&boxModel);
        comboBox.setView(&boxModelView);
        comboBox.setInsertPolicy(QComboBox::NoInsert);
        comboBox.setEditable(false);
        comboBox.setEnabled(false);
        comboBox.setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);
        QObject::connect(
                &comboBox,
                QOverload<int>::of(&QComboBox::currentIndexChanged),
                [this](int index) { loadParameters(index); });
        layout.addWidget(&comboBox, 1);

        applyButton.setIcon(
                QIcon(":/QtPdWidgets/images/apply-parameters.svg"));
        applyButton.setEnabled(false);
        QObject::connect(
                &applyButton, &QPushButton::clicked,
                [this]() { applyParameters(); });
        layout.addWidget(&applyButton, 0);

        QObject::connect(
                &watcher, &QFileSystemWatcher::fileChanged,
                [this](const QString &) {
                loadParameters(comboBox.currentIndex());
                });
        layout.setSizeConstraint(QLayout::SetMinimumSize);
    }

    ~Impl() {
        if (not parameters.isEmpty()) {
            foreach (Parameter *par, parameters) {
                delete par;
            }
            parameters.clear();
        }
        QObject::disconnect(&watcher, &QFileSystemWatcher::fileChanged,
                nullptr, nullptr);
        if (not currentPath.isEmpty()) {
            watcher.removePath(currentPath);
        }
    }

    void loadDialog()
    {

        QFileDialog dialog(widget);
        dialog.setAcceptMode(QFileDialog::AcceptOpen);
        dialog.setFileMode(QFileDialog::Directory);
        dialog.setOption(QFileDialog::ShowDirsOnly, true);

        int ret = dialog.exec(); // FIXME replace exec()

        if (ret != QDialog::Accepted) {
            return;
        }

        widget->setPath(dialog.selectedFiles()[0]);
    }

    void connectParameters()
    {
        foreach (Parameter *par, parameters) {
            QUrl parameterUrl(par->url);
            QtPdCom::Process *process(nullptr);
            foreach (QtPdCom::Process *proc, processes) {
                if (parameterUrl.host() == proc->getPeerName()) {
                    process = proc;
                    // FIXME check port
                    break;
                }
            }
            if (process) {
                par->connectParameter(process);
            }
            else {
                qWarning() << "No process found for" << parameterUrl;
            }
        }
        for (const auto process : processes)
            process->callPendingCallbacks();
    }

    void applyParameters()
    {
        foreach (Parameter *par, parameters) {
            par->writeValue(par->getSavedValue());
        }
    }

    void loadParameters(int index)
    {
        QObject::disconnect(&watcher, &QFileSystemWatcher::fileChanged,
                nullptr, nullptr);
        watcher.removePath(currentPath);

        if (index < 0) {
            return;
        }

        if (currentPath.isEmpty()) {
            currentPath = path;
            applyButton.setEnabled(false);
            return;
        }

        currentPath = QDir(path).filePath(comboBox.currentText());

        QFile file(currentPath);
        if (!file.open(QIODevice::ReadOnly)) {
            qWarning() << tr("Failed to open %1.").arg(currentPath);
            return;
        }

        QByteArray ba = file.readAll();
        file.close();

        QJsonParseError err;
        QJsonDocument doc(QJsonDocument::fromJson(ba, &err));

        if (err.error != QJsonParseError::NoError) {
            qCritical() << "Parameter file parsing error (" << err.error
                << ") at offset " << err.offset << ": "
                << err.errorString();
            return;
        }

        QJsonObject layoutObject(doc.object());
        QJsonArray parameterArray(layoutObject["parameters"].toArray());

        clearParameters();
        fromJson(parameterArray);

        connectParameters();
        compareValues();

        watcher.addPath(currentPath);
        QObject::connect(
                &watcher, &QFileSystemWatcher::fileChanged,
                [this, index](const QString &) { loadParameters(index); });
    }

    void updateProcessPixmap()
    {
        QString image;
        if (different) {
            image = ":/QtPdWidgets/images/parameters-differing.svg";
        }
        else {
            image = ":/QtPdWidgets/images/parameters-equal.svg";
        }
        applyButton.setEnabled(different);
        processPixmap.load(image);
        label.setPixmap(
                processPixmap.scaledToHeight(30, Qt::SmoothTransformation));
    }

    void compareValues()
    {
        different = false;
        foreach (Parameter *par, parameters) {
#ifdef DEBUG_PD_PARAMETERSETWIDGET
            qDebug() << __func__ << par->url << par->hasData()
                << par->getCurrentValue() << par->getSavedValue();
#endif
            if (not par->hasData()
                    or par->getCurrentValue() != par->getSavedValue()) {
                different = true;
                break;
            }
        }
        updateProcessPixmap();
    }


    void fromJson(const QJsonArray &array)
    {
        foreach (QJsonValue arrayValue, array) {
            Parameter *parameter = new Parameter(widget);
            parameter->fromJson(arrayValue);
            parameters.append(parameter);

#ifdef DEBUG_PD_PARAMETERSETWIDGET
            qDebug() << __func__ << "loaded" << parameter->url;
#endif

            QObject::connect(parameter, &Parameter::dataChanged,
                    [this]() { compareValues(); });
        }
    }

    void clearParameters()
    {
        foreach (Parameter *par, parameters) {
            delete par;
        }
        parameters.clear();
        for (const auto process : processes)
            process->callPendingCallbacks();
    }

    ParameterSetWidget * const widget;

    QHBoxLayout layout;
    QLabel label;
    QComboBox comboBox;
    QPushButton applyButton;

    QList<Pd::Parameter *> parameters;
    QPixmap processPixmap;
    QFileSystemModel boxModel;
    QListView boxModelView;

    QString path; /**< Directory path. */
    QSet<QtPdCom::Process *> processes;
    QString currentPath;
    QFileSystemWatcher watcher;
    bool different;
};

/****************************************************************************/

/** Constructor.
 */
ParameterSetWidget::ParameterSetWidget(
        QWidget *parent /**< parent widget */):
    QFrame(parent),
    impl(std::unique_ptr<Impl>(new Impl(this)))
{
    setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Fixed);
}

/****************************************************************************/

/** Destructor.
 */
ParameterSetWidget::~ParameterSetWidget()
{
}

/****************************************************************************/

void ParameterSetWidget::setProcesses(QSet<QtPdCom::Process *> processes)
{
    impl->processes = processes;
    impl->connectParameters();
}

/****************************************************************************/

void ParameterSetWidget::setPath(const QString &path)
{
    impl->currentPath.clear();
    impl->path = path;
    impl->boxModel.setRootPath(path);
    impl->boxModelView.setRootIndex(impl->boxModel.index(path));
    impl->comboBox.setEnabled(true);
}

/****************************************************************************/

const QString &ParameterSetWidget::getPath() const
{
    return impl->path;
}

/****************************************************************************/

bool ParameterSetWidget::event(QEvent *event)
{
    if (event->type() == QEvent::ContextMenu) {
        QContextMenuEvent *menuEvent =
            static_cast<QContextMenuEvent *>(event);

        QMenu menu;

        QAction *action;

        action = new QAction(this);
        action->setText(tr("Change Folder..."));
        action->setIcon(QIcon(":/QtPdWidgets/images/document-open.svg"));
        connect(action, &QAction::triggered,
                this, [this]() { impl->loadDialog(); });
        menu.addAction(action);

        menu.exec(menuEvent->globalPos());

        return true;
    }

    return QFrame::event(event);
}

/****************************************************************************/

QSize ParameterSetWidget::sizeHint() const
{
    return QSize(100, 30);
}

/****************************************************************************/
